unit DSList;

{  Display Summary via Memo component
   Copyright (c) 1996, by Philip Stevenson
}

interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  SysUtils, StdCtrls, Messages, ExtCtrls,
  Dialogs, Dscomp, FileInfo;

type

  TSummaryDlg = class(TForm)
    SummaryMemo: TMemo;
    Panel1: TPanel;
    PrintBtn: TBitBtn;
    SaveBtn: TBitBtn;
    CopyBtn: TBitBtn;
    SaveDialog1: TSaveDialog;
    procedure FormShow(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure PrintBtnClick(Sender: TObject);
    procedure SaveBtnClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure CopyBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
    SaveName: string;
    InList: TStringList;
    procedure SendFileOut(const FN: string; const print: boolean);
  public
    { Public declarations }
    procedure ShowSummary;
    procedure LoadMemo(SL: TSourceList; const SFile: string;
      const line: Integer; const len: Integer);
    procedure AddToSum(const S: string);
    procedure FileSum(const Fin1, Fin2: TFileInfo);
    procedure BuildSummary(SL1, SL2: TSourceList;
  const Altered: boolean; const MatchLines: Integer);
    procedure ShowHelp;
  end;

var
  SummaryDlg: TSummaryDlg;

implementation

{$R *.DFM}

{TSummaryDlg methods}

procedure TSummaryDlg.FormShow(Sender: TObject);
begin
  ActiveControl := SummaryMemo;
end;

procedure TSummaryDlg.OKBtnClick(Sender: TObject);
begin
  Close
end;

procedure TSummaryDlg.FormCreate(Sender: TObject);
begin
  Height := 440;
  Width := 500;
  Left := Screen.Width - Width - 20; {put near to right}
  Top := Screen.Height - Height - 25; {put near to bottom}
  InList := TStringList.Create;
end;

procedure TSummaryDlg.FormDestroy(Sender: TObject);
begin
  InList.Free;
end;

procedure TSummaryDlg.AddToSum(const S: string);
begin
  InList.Add(S)
end;

procedure TSummaryDlg.FileSum(const Fin1, Fin2: TFileInfo);
var
  S: string;
  diftime: TDateTime;
begin
  with InList do
  begin
    Clear;
    Add('File Information:');
    if Fin1.FileName = Fin2.FileName then
    begin
      Add(Fin1.FileName);
      if Fin1.FilePath = Fin2.FilePath then
        Add(' File is compared to itself?')
      else
        Add(' Files are in different directories.');
    end
    else
      Add(Fin1.FileName+' compared to '+ Fin2.FileName);

    diftime := Fin1.FileTime-Fin2.FileTime;
    if diftime > 0 then
      S := Fin1.FileName+' is newer by '+TimeDiffStr(diftime)
    else if diftime < 0 then
      S := Fin1.FileName+' is older by '+TimeDiffStr(Abs(diftime))
    else
      S := 'Files have same date and time.';
    Add(S);
    Add(Fin1.ShowFileStats);
    Add('Current File Path: '+Fin1.FilePath);
    Add(Fin2.ShowFileStats);
    if Fin1.FilePath = Fin2.FilePath then
      Add('Previous File Path is same')
    else
      Add('Previous File Path: '+Fin2.FilePath);
    Add('----');
  end;
end;

procedure TSummaryDlg.BuildSummary(SL1, SL2: TSourceList;
  const Altered: boolean; const MatchLines: Integer);
var
  pc: real;

  procedure ShowFileInfo(const dstr: string; SL: TSourceList);
  begin
    with InList do
    begin
      Add(' ');
      Add(dstr + ExtractFileName(SL.SourceFile));
      Add('Lines in this file: '+IntToStr(SL.Count));
      if Altered then
      begin
        Add('First Line not matching: '+SL.FirstMismatch);
        Add(' Last Line not matching: '+SL.LastMismatch);
      end;
    end
  end;

begin
  with InList do
  begin
    if Altered then
    begin
      Add(' Matching Lines: '+IntToStr(MatchLines));
      pc := (100.0 * MatchLines) / SL1.Count;
      Add(Format(' Percent Lines Matching in current file: %2.1f', [pc]));
      if pc < 20.0 then
        Add('THESE TWO FILES SEEM TO BE UNRELATED!')
      else if pc > 95.0 then
        Add('Files are closely matched.');
      if (SL1.Count > SL2.Count) or (SL1.Count < SL2.Count) then
        Add(Format('Lines total change (+/-): %2d',[SL1.Count - SL2.Count]))
    end;
    ShowFileInfo('Current Text File: ', SL1);
    ShowFileInfo('Previous Text File: ', SL2);
    Add('Comments:');
  end;
end;

procedure TSummaryDlg.ShowSummary;
{-Generate Summary Report}
begin
  Caption := 'Summary Report';
  SummaryMemo.Lines :=  InList;
  SaveName := 'Summary';
  Show;
end;

procedure TSummaryDlg.LoadMemo(SL: TSourceList; const SFile: string;
 const line: Integer; const len: Integer);
{-Summary Memo is "borrowed" to display DFM (or other source)}
var
  Fin: TFileInfo;
  ir: Integer;
begin
  Screen.Cursor := crHourGlass;
  SummaryMemo.Clear;
  ActiveControl := SummaryMemo;
  Fin := TFileInfo.Create(SL.SourceFile);
  try
    SaveName := Fin.FileName; {save name}
    if CompareText(ExtractFileName(SFile), Fin.FileName) <> 0 then
      Caption := 'Listing for:  '+SFile+' (text for '+Fin.FileName+')'
    else
      Caption := 'Listing for:  '+SFile;
    with SummaryMemo.Lines do
    try
      Add(Fin.ShowFileStats);
      Add('Path: '+Fin.FilePath);
      Add('Lines in '+SFile+': '+IntToStr(SL.Count));
      Add('-----------------------------------');
      for ir := 0 to SL.Count-1 do
        SummaryMemo.Lines.Add(Format('%3d: %s', [ir+1, SL.Strings[ir]]));
    except
      Screen.Cursor := crDefault;
      MessageBeep(MB_ICONASTERISK);
      MessageDlg('Unable to load all of file '#13+SFile, mtError, [mbOk], 0);
    end;
    {move cursor to selected line}
    with SummaryMemo do
    begin
      SelStart := SendMessage(Handle, EM_LINEINDEX, line+3, 0);
      SelLength := len+5;
      SendMessage(Handle, EM_SETSEL, 0, MakeLong(SelStart, SelStart+SelLength));
    end;
    SaveName := SFile;
    Show;
  finally
    Screen.Cursor := crDefault;
    Fin.Free;
  end;
end;

procedure TSummaryDlg.SendFileOut(const FN: string; const print: boolean);
var
  PF: TextFile;
  ir: word;
begin
  AssignFile(PF, FN);
  try
    Rewrite(PF);                            { open printer }
    with SummaryMemo.Lines do
      try
        writeln(PF, 'Summary as of: ',
          FormatDateTime(DATE_TIME_FORMAT, Now));
        writeln(PF);
        writeln(PF);
        for ir := 0 to Count-1 do
          writeln(PF, Strings[ir]);
        if print then
          write(PF, ^L); {form feed}
      finally
        CloseFile(PF);                        { close printer }
      end;
  except
    on EInOutError do
      MessageDlg('Error on output of summary to '+FN, mtError, [mbOk], 0);
  end;
end;

procedure TSummaryDlg.PrintBtnClick(Sender: TObject);
{-Prints all of the strings in SummaryMemo }
begin
(*  AssignPrn(PF);                    { assign Prn to printer } *)
  SendFileOut('PRN', True);    { assign Prn to printer }
end;

procedure TSummaryDlg.SaveBtnClick(Sender: TObject);
{-Save selected text to Clipboard }
begin
  with SaveDialog1 do
  begin
    FileName := ChangeFileExt(SaveName, '.lst');
    if execute then
      SendFileOut(FileName, False);
  end
end;

procedure TSummaryDlg.FormActivate(Sender: TObject);
begin
  with SummaryMemo do
    SendMessage(Handle, EM_SETSEL, 0, MakeLong(SelStart, SelStart+SelLength));
end;

procedure TSummaryDlg.CopyBtnClick(Sender: TObject);
{-Copy selected text to Clipboard }
begin
  with SummaryMemo do
    if SelLength <> 0 then
      CopyToClipboard
    else
      ShowMessage('Select a block of text to copy.');
end;

procedure TSummaryDlg.ShowHelp;
{-Summary Memo is "borrowed" to display Help}
const
  HL = 21;
  HelpText: array[1..HL] of string[80] = (
  'How to use...',
  ' ',
  '1. Options:',
  '   Choose line comparision and output display options.',
  ' ',
  '2. File Selection:',
  '   Select files by dragging them from File Manager or',
  '   Explorer, or click on Open Files to select files to',
  '   compare.  The second file will automatically be',
  '   selected if ''Autoload Second File'' is on.',
  '   Form files can also be compared.',
  ' ',
  '3. Doing Comparisons:',
  '   Click Compare to generate the difference list.',
  ' ',
  '4. Getting output report(s):',
  '   Click on Summary to get a summary report.',
  '   The Difference Grid can be copied to printer',
  '   or a file, or the Clipboard.',
  '   By right clicking on a line number in the Grid you',
  '   can display the corresponding source file area.');

var
  ir: Integer;
begin
  Screen.Cursor := crHourGlass;
  try
    SummaryMemo.Clear;
    ActiveControl := SummaryMemo;
    Caption := 'Help...';
    try
      for ir := 1 to HL do
        SummaryMemo.Lines.Add(Helptext[ir]);
    except
      MessageDlg('Unable to load help text', mtError, [mbOk], 0);
    end;
    SaveName := 'SDifHelp';
    Show;
  finally
    Screen.Cursor := crDefault;
  end;
end;

end.
